/*===========================================================================
  Copyright (C) 2012 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  This library is free software; you can redistribute it and/or modify it 
  under the terms of the GNU Lesser General Public License as published by 
  the Free Software Foundation; either version 2.1 of the License, or (at 
  your option) any later version.

  This library is distributed in the hope that it will be useful, but 
  WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License 
  along with this library; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  See also the full LGPL text here: http://www.gnu.org/copyleft/lesser.html
===========================================================================*/

package net.sf.okapi.filters.drupal.ui;

import net.sf.okapi.common.LocaleId;
import net.sf.okapi.common.Util;
import net.sf.okapi.common.ui.Dialogs;
import net.sf.okapi.common.ui.OKCancelPanel;
import net.sf.okapi.common.ui.UIUtil;
import net.sf.okapi.filters.drupal.IProjectEditor;
import net.sf.okapi.filters.drupal.Project;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;

public class ProjectDialog implements IProjectEditor {
	
	private Shell shell;
	private Project project;
	private ProjectTableModel tableMod;
	private SelectionAdapter CloseActions;
	private boolean result;
	private Text edHost;
	private Text edUser;
	private Text edPassword;
	private Text edSource;
	private Text edTarget;
	private Button chkNeutralLikeSource;

	public ProjectDialog () {
		// Needed to be able to instantiate this class with Class.forName().
	}
	
	private void create (Shell parent,
		boolean inProcess)
	{
		result = false;
		shell = new Shell(parent, SWT.CLOSE | SWT.TITLE | SWT.RESIZE | SWT.APPLICATION_MODAL);
		shell.setText("Drupal Project");
		UIUtil.inheritIcon(shell, parent);
		shell.setLayout(new GridLayout());
		
		Composite cmpTmp = new Composite(shell, SWT.BORDER);
		cmpTmp.setLayout(new GridLayout(4, false));
		cmpTmp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Label label = new Label(cmpTmp, SWT.NONE);
		label.setText("Host:");
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		edHost = new Text(cmpTmp, SWT.BORDER);
		edHost.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));

		label = new Label(cmpTmp, SWT.NONE);
		label.setText("Source:");
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		edSource = new Text(cmpTmp, SWT.BORDER);
		edSource.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));
		edSource.setEditable(false);
		
		label = new Label(cmpTmp, SWT.NONE);
		label.setText("User name:");
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		edUser = new Text(cmpTmp, SWT.BORDER);
		edUser.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));
		
		label = new Label(cmpTmp, SWT.NONE);
		label.setText("Target:");
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		edTarget = new Text(cmpTmp, SWT.BORDER);
		edTarget.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));
		edTarget.setEditable(false);
		
		label = new Label(cmpTmp, SWT.NONE);
		label.setText("Password:");
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_END));
		edPassword = new Text(cmpTmp, SWT.BORDER);
		edPassword.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.FILL_HORIZONTAL));
		edPassword.setEchoChar('*');
		
		chkNeutralLikeSource = new Button(cmpTmp, SWT.CHECK);
		chkNeutralLikeSource.setText("Treat neutral (und) entries like source entries");
		GridData gdTmp = new GridData(GridData.FILL_HORIZONTAL);
		gdTmp.horizontalSpan = 2;
		chkNeutralLikeSource.setLayoutData(gdTmp);
		
		Button btRefresh = UIUtil.createGridButton(shell, SWT.PUSH, "Refresh Resources List", UIUtil.BUTTON_DEFAULT_WIDTH*2, 1);
		btRefresh.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				refresh();
			};
		});

		cmpTmp = new Composite(shell, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		cmpTmp.setLayout(layout);
		cmpTmp.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		final Table tableDocs = new Table(cmpTmp, SWT.CHECK | SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION | SWT.V_SCROLL);
		gdTmp = new GridData(GridData.FILL_BOTH);
		gdTmp.minimumHeight = 300;
		gdTmp.minimumWidth = 550;
		tableDocs.setLayoutData(gdTmp);
		tableDocs.setHeaderVisible(true);
		tableDocs.setLinesVisible(true);
		tableMod = new ProjectTableModel();
		tableMod.initializeTable(tableDocs);
		
		//--- Dialog-level buttons

		CloseActions = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if ( e.widget.getData().equals("h") ) { //$NON-NLS-1$
					// Don't use context because this dialog box may be used from anywhere
					Util.openWikiTopic("Drupal Filter");
					return;
				}
				if ( e.widget.getData().equals("o") ) { //$NON-NLS-1$
					saveData();
					project.save();
					result = true;
				}
				shell.close();
			};
		};
		OKCancelPanel pnlActions = new OKCancelPanel(shell, SWT.NONE, CloseActions, true,
			(inProcess ? "Continue" : "OK"));
		pnlActions.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		shell.setDefaultButton(btRefresh);

		shell.pack();
		shell.setMinimumSize(shell.getSize());
		Point startSize = shell.getMinimumSize();
		if ( startSize.x < 450 ) startSize.x = 450;
		shell.setSize(startSize);

		Dialogs.centerWindow(shell, parent);
	}
	
	private void setData (Project project) {
		this.project = project;
		edHost.setText(project.getHost());
		edUser.setText(project.getUser());
		edPassword.setText(project.getPassword());
		edSource.setText(project.getSourceLocale().toString());
		edTarget.setText(project.getTargetLocale().toString());
		chkNeutralLikeSource.setSelection(project.getNeutralLikeSource());
	}
	
	private boolean saveData () {
		String tmp = edHost.getText().trim();
		if ( tmp.isEmpty() ) {
			return false;
		}
		project.setHost(tmp);

		tmp = edUser.getText().trim();
		if ( tmp.isEmpty() ) {
			return false;
		}
		project.setUser(tmp);
		
		tmp = edPassword.getText().trim();
		if ( tmp.isEmpty() ) {
			return false;
		}
		project.setPassword(tmp);
		
		tmp = edSource.getText().trim();
		if ( tmp.isEmpty() ) {
			return false;
		}
		project.setSourceLocale(LocaleId.fromString(tmp));
		
		tmp = edTarget.getText().trim();
		if ( tmp.isEmpty() ) {
			return false;
		}
		project.setTargetLocale(LocaleId.fromString(tmp));
		
		project.setNeutralLikeSource(chkNeutralLikeSource.getSelection());
		
		tableMod.saveData();
		return true;
	}
	
	private boolean showDialog (Project project) {
		tableMod.setProject(project);
		setData(project);
		shell.open();
		while ( !shell.isDisposed() ) {
			if ( !shell.getDisplay().readAndDispatch() )
				shell.getDisplay().sleep();
		}
		return result;
	}

	@Override
	public boolean edit (Object parent,
		Project project,
		boolean inProcess)
	{
		Shell shell = null;
		if (( parent != null ) && ( parent instanceof Shell )) {
			shell = (Shell)parent;
		}
		create(shell, inProcess);
		boolean res = showDialog(project);
		
		return res;
	}

	private void refresh () {
		try {
			if ( !saveData() ) return;
			project.refreshEntries(false);
			tableMod.setProject(project);
		}
		catch ( Throwable e ) {
			Dialogs.showError(shell, "Error during the refresh:\n"+e.getMessage(), null);
		}
	}

}
