/*===========================================================================
  Copyright (C) 2008-2011 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  This library is free software; you can redistribute it and/or modify it 
  under the terms of the GNU Lesser General Public License as published by 
  the Free Software Foundation; either version 2.1 of the License, or (at 
  your option) any later version.

  This library is distributed in the hope that it will be useful, but 
  WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser 
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License 
  along with this library; if not, write to the Free Software Foundation, 
  Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

  See also the full LGPL text here: http://www.gnu.org/copyleft/lesser.html
===========================================================================*/

package net.sf.okapi.filters.openoffice;

import static org.junit.Assert.*;

import java.io.File;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import net.sf.okapi.common.Event;
import net.sf.okapi.common.TestUtil;
import net.sf.okapi.common.annotation.GenericAnnotationType;
import net.sf.okapi.common.filters.FilterConfiguration;
import net.sf.okapi.common.filters.FilterTestDriver;
import net.sf.okapi.common.filters.InputDocument;
import net.sf.okapi.common.filters.RoundTripComparison;
import net.sf.okapi.common.filterwriter.GenericContent;
import net.sf.okapi.common.resource.Code;
import net.sf.okapi.common.resource.ITextUnit;
import net.sf.okapi.common.resource.RawDocument;
import net.sf.okapi.common.LocaleId;
import net.sf.okapi.filters.openoffice.ODFFilter;

import org.junit.Before;
import org.junit.Test;

public class ODFFilterTest {
	
	private ODFFilter filter;
	private String root;
	private LocaleId locEN = LocaleId.fromString("en");
	private GenericContent fmt;

	@Before
	public void setUp() {
		filter = new ODFFilter();
		root = TestUtil.getParentDir(this.getClass(), "/TestDocument01.odt_content.xml");
		fmt = new GenericContent();
	}
	
	@Test
	public void testFirstTextUnit () {
		ITextUnit tu = FilterTestDriver.getTextUnit(filter,
			new InputDocument(root+"TestDocument01.odt_content.xml", null),
			"UTF-8", locEN, locEN, 1);
		assertNotNull(tu);
		assertEquals("Heading 1", tu.getSource().toString());
	}

	@Test
	public void testITSMarkup () {
		List<Event> events = getEvents(root+"Content_WithITS.xml", LocaleId.ENGLISH);
		
		// With translate='no'
		ITextUnit tu = FilterTestDriver.getTextUnit(events, 2);
		assertEquals("To translate <1>NOT to translate <2>but translate this</2></1>.",
			fmt.setContent(tu.getSource().getFirstContent()).toString());
		List<Code> codes = tu.getSource().getFirstContent().getCodes();
		// Don't translate (code 0 is start tag
		assertEquals(false, codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.TRANSLATE)
			.getBoolean(GenericAnnotationType.TRANSLATE_VALUE));
		// Translate (code 1 is start tag
		assertEquals(true, codes.get(1).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.TRANSLATE)
			.getBoolean(GenericAnnotationType.TRANSLATE_VALUE));
		
		// Localization notes
		tu = FilterTestDriver.getTextUnit(events, 3);
		assertEquals("Text with <1>a note</1>. More text <2>with a note <3>and another note</3></2>.",
			fmt.setContent(tu.getSource().getFirstContent()).toString());
		codes = tu.getSource().getFirstContent().getCodes();
		// First note (code 0 is start tag)
		assertEquals("Localization Note", codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_VALUE));
		assertEquals("description", codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_TYPE));
		// Second note (code 2 is start tag)
		assertEquals("Localization Node Outer", codes.get(2).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_VALUE));
		assertEquals("alert", codes.get(2).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_TYPE));
		// Third note (code 3 is start tag)
		assertEquals("Localization Node Inner", codes.get(3).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_VALUE));
		assertEquals("description", codes.get(3).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCNOTE)
			.getString(GenericAnnotationType.LOCNOTE_TYPE));
		
		// Terminology + translate
		tu = FilterTestDriver.getTextUnit(events, 5);
		codes = tu.getSource().getFirstContent().getCodes();
		assertEquals("Text with <1>a very long term <2>made of several words</2></1> and more text.",
			fmt.setContent(tu.getSource().getFirstContent()).toString());
		// Term (code 0 is start tag)
		assertNotNull(codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.TERM));
		assertEquals(0.8, codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.TERM)
			.getDouble(GenericAnnotationType.TERM_CONFIDENCE), 0.0);
		assertEquals("abc", codes.get(0).getGenericAnnotationString(
			GenericAnnotationType.TERM, GenericAnnotationType.ANNOTATORREF));
		// Translate (code 1 is start tag)
		assertEquals(false, codes.get(1).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.TRANSLATE)
			.getBoolean(GenericAnnotationType.TRANSLATE_VALUE));

		// Locale filter
		tu = FilterTestDriver.getTextUnit(events, 6);
		codes = tu.getSource().getFirstContent().getCodes();
		assertEquals("Locale filter: <1>for FR</1> and <2>Not for FR</2>.",
			fmt.setContent(tu.getSource().getFirstContent()).toString());
		// Applies to fr (code 0 is start tag)
		assertEquals("fr", codes.get(0).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCFILTER)
			.getString(GenericAnnotationType.LOCFILTER_VALUE));
		// Applies to fr (code 2 is start tag)
		assertEquals("!fr", codes.get(2).getGenericAnnotations().getFirstAnnotation(GenericAnnotationType.LOCFILTER)
			.getString(GenericAnnotationType.LOCFILTER_VALUE));
	}
	
	@Test
	public void testDefaultInfo () {
		assertNotNull(filter.getParameters());
		assertNotNull(filter.getName());
		List<FilterConfiguration> list = filter.getConfigurations();
		assertNotNull(list);
		assertTrue(list.size()>0);
	}
	
	@Test
	public void testDoubleExtraction () throws URISyntaxException {
		ArrayList<InputDocument> list = new ArrayList<InputDocument>();
		list.add(new InputDocument(root+"TestDocument01.odt_content.xml", null));
		list.add(new InputDocument(root+"TestDocument01.odt_meta.xml", null));
		list.add(new InputDocument(root+"TestDocument01.odt_styles.xml", null));
		list.add(new InputDocument(root+"TestDocument02.odt_content.xml", null));
		list.add(new InputDocument(root+"ODFTest_footnote.xml", null));
		list.add(new InputDocument(root+"TestSpreadsheet01.ods_content.xml", null));
		
		RoundTripComparison rtc = new RoundTripComparison();
		assertTrue(rtc.executeCompare(filter, list, "UTF-8", locEN, locEN));
	}

	private ArrayList<Event> getEvents(String filePath,
		LocaleId srcLang)
	{
		ArrayList<Event> list = new ArrayList<Event>();
		File file = new File(filePath);
		filter.open(new RawDocument(file.toURI(), "UTF-8", srcLang));
		while ( filter.hasNext() ) {
			Event event = filter.next();
			list.add(event);
		}
		filter.close();
		return list;
	}

}
